package cn.iocoder.foodnexus.module.order.controller.app.customerOrder;

import cn.iocoder.foodnexus.framework.common.enums.UserSystemEnum;
import cn.iocoder.foodnexus.framework.common.pojo.CommonResult;
import cn.iocoder.foodnexus.framework.common.pojo.PageResult;
import cn.iocoder.foodnexus.framework.common.util.CommonUtil;
import cn.iocoder.foodnexus.framework.common.util.object.BeanUtils;
import cn.iocoder.foodnexus.module.erp.dal.dataobject.sale.ErpSaleOutDO;
import cn.iocoder.foodnexus.module.erp.service.sale.ErpSaleOutService;
import cn.iocoder.foodnexus.module.order.controller.admin.customerorder.vo.CustomerOrderPageReqVO;
import cn.iocoder.foodnexus.module.order.controller.admin.customerorder.vo.CustomerOrderRecordMapVO;
import cn.iocoder.foodnexus.module.order.controller.admin.customerorder.vo.CustomerOrderRespVO;
import cn.iocoder.foodnexus.module.order.controller.admin.customerorderitem.vo.CustomerOrderItemRespVO;
import cn.iocoder.foodnexus.module.order.controller.app.customerOrder.vo.*;
import cn.iocoder.foodnexus.module.order.dal.dataobject.customerorder.CustomerOrderDO;
import cn.iocoder.foodnexus.module.order.dal.dataobject.customerorderitem.CustomerOrderItemDO;
import cn.iocoder.foodnexus.module.order.dto.AppCustomerReturnOrderReqVO;
import cn.iocoder.foodnexus.module.order.service.customerorder.CustomerOrderService;
import cn.iocoder.foodnexus.module.order.service.customerorderitem.CustomerOrderItemService;
import cn.iocoder.foodnexus.module.order.service.customerorderrecord.CustomerOrderRecordService;
import cn.iocoder.foodnexus.module.system.annotations.AppSystemAuth;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import jakarta.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.sql.rowset.serial.SerialException;
import java.util.List;
import java.util.Map;

import static cn.iocoder.foodnexus.framework.common.pojo.CommonResult.success;
import static cn.iocoder.foodnexus.framework.security.core.util.SecurityFrameworkUtils.getLoginUserId;

@Tag(name = "APP - 客户总订单")
@RestController
@RequestMapping("/order/customer-order")
@Validated
@AppSystemAuth(UserSystemEnum.CUSTOMER)
public class AppCustomerOrderController {

    @Resource
    private CustomerOrderService customerOrderService;
    @Autowired
    private CustomerOrderRecordService customerOrderRecordService;
    @Autowired
    private CustomerOrderItemService customerOrderItemService;

    @GetMapping("status-count")
    @Operation(summary = "客户订单各状态对应数量")
    public CommonResult<Map<String, Long>> queryStatusCount() {
        Long loginUserId = getLoginUserId();
        return success(customerOrderService.queryStatusCount(loginUserId));
    }

    @PostMapping("/create")
    @Operation(summary = "创建客户总订单")
    public CommonResult<Long> createCustomerOrder(@Valid @RequestBody AppCustomerOrderSaveReqVO createReqVO) {
        return success(customerOrderService.appCreateCustomerOrder(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新客户总订单")
    public CommonResult<Boolean> updateCustomerOrder(@Valid @RequestBody AppCustomerOrderSaveReqVO updateReqVO) throws SerialException {
        if (CommonUtil.isEmpty(updateReqVO.getId())) {
            throw new SerialException("订单id不能为空");
        }
        customerOrderService.appUpdateCustomerOrder(updateReqVO);
        return success(true);
    }

    @PostMapping("cancel")
    @Operation(summary = "取消客户订单")
    public CommonResult<Boolean> cancel(@Valid @RequestBody AppCustomerOrderRemarkReqVO reqVO) {
        customerOrderService.cancel(reqVO);
        return success(Boolean.TRUE);
    }

    @GetMapping("receipt")
    @Operation(summary = "获取签收客户订单信息")
    @Parameter(name = "id", description = "客户订单id", required = true)
    public CommonResult<List<AppCustomerOrderReceiptRespVO>> receipt(@RequestParam("id") Long id) {
        return success(customerOrderService.queryReceipt(id));
    }

    @PostMapping("receipt")
    @Operation(summary = "签收客户订单")
    public CommonResult<Boolean> receipt(@Valid @RequestBody AppCustomerOrderRemarkReqVO reqVO) {
        customerOrderService.receipt(reqVO);
        return success(Boolean.TRUE);
    }

    @PostMapping("return")
    @Operation(summary = "退货")
    public CommonResult<Boolean> orderReturn(@Valid @RequestBody AppCustomerReturnOrderReqVO reqVO) {
        customerOrderService.orderReturn(reqVO);
        return success(Boolean.TRUE);
    }

    @GetMapping("query-return")
    @Operation(summary = "查询退货说明")
    @Parameter(name = "id", description = "客户订单id", required = true)
    public CommonResult<AppCustomerOrderReturnRespVO> queryReturn(@RequestParam("id") Long id) {
        return success(customerOrderService.queryReturn(id));
    }

    @GetMapping("score")
    @Operation(summary = "查询订单评价")
    @Parameter(name = "id", description = "客户订单id", required = true)
    public CommonResult<AppCustomerOrderScoreRespVO> queryScore(@RequestParam("id") Long id) {
        return success(customerOrderService.queryScore(id));
    }

    @PostMapping("score")
    @Operation(summary = "评价订单")
    public CommonResult<Boolean> score(@Valid @RequestBody AppCustomerOrderScoreReqVO reqVO) {
        customerOrderService.score(reqVO);
        return success(Boolean.TRUE);
    }

    @DeleteMapping("/delete-list")
    @Parameter(name = "ids", description = "编号", required = true)
    @Operation(summary = "批量删除客户总订单")
    public CommonResult<Boolean> deleteCustomerOrderList(@RequestParam("ids") List<Long> ids) {
        customerOrderService.deleteCustomerOrderListByIds(ids);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得客户总订单")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    public CommonResult<CustomerOrderRespVO> getCustomerOrder(@RequestParam("id") Long id) {
        CustomerOrderDO customerOrder = customerOrderService.getCustomerOrder(id);
        return success(BeanUtils.toBean(customerOrder, CustomerOrderRespVO.class, item -> {
            item.setCurrentRecords(customerOrderRecordService.queryCurrentRecords(item.getId(), item.getOrderStatus()));
            item.setOrderItems(BeanUtils.toBean(customerOrderItemService.queryByOrderId(item.getId()), CustomerOrderItemRespVO.class));
        }));
    }

    @GetMapping("/page")
    @Operation(summary = "获得客户总订单分页")
    public CommonResult<PageResult<CustomerOrderRespVO>> getCustomerOrderPage(@Valid CustomerOrderPageReqVO pageReqVO) {
        pageReqVO.setCreator(getLoginUserId());
        PageResult<CustomerOrderDO> pageResult = customerOrderService.getCustomerOrderPage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, CustomerOrderRespVO.class, item -> {
            item.setCurrentRecords(customerOrderRecordService.queryCurrentRecords(item.getId(), item.getOrderStatus()));
            item.setProductInfos(CommonUtil.listConvert(customerOrderItemService.queryByOrderId(item.getId()), CustomerOrderItemDO::getProductInfo));
        }));
    }

    @GetMapping("/record")
    @Operation(summary = "订单进度")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    public CommonResult<List<CustomerOrderRecordMapVO>> queryRecord(@RequestParam("id") Long id) {
        return success(customerOrderRecordService.queryRecordMap(id));
    }

}