package cn.iocoder.foodnexus.module.order.controller.admin.customerorderitem;

import org.springframework.web.bind.annotation.*;
import jakarta.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import jakarta.validation.constraints.*;
import jakarta.validation.*;
import jakarta.servlet.http.*;
import java.util.*;
import java.io.IOException;

import cn.iocoder.foodnexus.framework.common.pojo.PageParam;
import cn.iocoder.foodnexus.framework.common.pojo.PageResult;
import cn.iocoder.foodnexus.framework.common.pojo.CommonResult;
import cn.iocoder.foodnexus.framework.common.util.object.BeanUtils;
import static cn.iocoder.foodnexus.framework.common.pojo.CommonResult.success;

import cn.iocoder.foodnexus.framework.excel.core.util.ExcelUtils;

import cn.iocoder.foodnexus.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.foodnexus.framework.apilog.core.enums.OperateTypeEnum.*;

import cn.iocoder.foodnexus.module.order.controller.admin.customerorderitem.vo.*;
import cn.iocoder.foodnexus.module.order.dal.dataobject.customerorderitem.CustomerOrderItemDO;
import cn.iocoder.foodnexus.module.order.service.customerorderitem.CustomerOrderItemService;

@Tag(name = "管理后台 - 客户订单-子订单")
@RestController
@RequestMapping("/order/customer-order-item")
@Validated
public class CustomerOrderItemController {

    @Resource
    private CustomerOrderItemService customerOrderItemService;

    @PostMapping("/create")
    @Operation(summary = "创建客户订单-子订单")
    @PreAuthorize("@ss.hasPermission('order:customer-order-item:create')")
    public CommonResult<Long> createCustomerOrderItem(@Valid @RequestBody CustomerOrderItemSaveReqVO createReqVO) {
        return success(customerOrderItemService.createCustomerOrderItem(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新客户订单-子订单")
    @PreAuthorize("@ss.hasPermission('order:customer-order-item:update')")
    public CommonResult<Boolean> updateCustomerOrderItem(@Valid @RequestBody CustomerOrderItemSaveReqVO updateReqVO) {
        customerOrderItemService.updateCustomerOrderItem(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除客户订单-子订单")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('order:customer-order-item:delete')")
    public CommonResult<Boolean> deleteCustomerOrderItem(@RequestParam("id") Long id) {
        customerOrderItemService.deleteCustomerOrderItem(id);
        return success(true);
    }

    @DeleteMapping("/delete-list")
    @Parameter(name = "ids", description = "编号", required = true)
    @Operation(summary = "批量删除客户订单-子订单")
                @PreAuthorize("@ss.hasPermission('order:customer-order-item:delete')")
    public CommonResult<Boolean> deleteCustomerOrderItemList(@RequestParam("ids") List<Long> ids) {
        customerOrderItemService.deleteCustomerOrderItemListByIds(ids);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得客户订单-子订单")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('order:customer-order-item:query')")
    public CommonResult<CustomerOrderItemRespVO> getCustomerOrderItem(@RequestParam("id") Long id) {
        CustomerOrderItemDO customerOrderItem = customerOrderItemService.getCustomerOrderItem(id);
        return success(BeanUtils.toBean(customerOrderItem, CustomerOrderItemRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得客户订单-子订单分页")
    @PreAuthorize("@ss.hasPermission('order:customer-order-item:query')")
    public CommonResult<PageResult<CustomerOrderItemRespVO>> getCustomerOrderItemPage(@Valid CustomerOrderItemPageReqVO pageReqVO) {
        PageResult<CustomerOrderItemDO> pageResult = customerOrderItemService.getCustomerOrderItemPage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, CustomerOrderItemRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出客户订单-子订单 Excel")
    @PreAuthorize("@ss.hasPermission('order:customer-order-item:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportCustomerOrderItemExcel(@Valid CustomerOrderItemPageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<CustomerOrderItemDO> list = customerOrderItemService.getCustomerOrderItemPage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "客户订单-子订单.xls", "数据", CustomerOrderItemRespVO.class,
                        BeanUtils.toBean(list, CustomerOrderItemRespVO.class));
    }

}