package cn.iocoder.foodnexus.module.customerpermission.core.db;

import cn.hutool.core.collection.CollUtil;
import cn.iocoder.foodnexus.module.customerpermission.core.rule.CustomerPermissionRule;
import cn.iocoder.foodnexus.module.customerpermission.core.rule.CustomerPermissionRuleFactory;
import cn.iocoder.foodnexus.framework.mybatis.core.util.MyBatisUtils;
import com.baomidou.mybatisplus.extension.plugins.handler.MultiDataPermissionHandler;
import lombok.RequiredArgsConstructor;
import net.sf.jsqlparser.expression.Expression;
import net.sf.jsqlparser.expression.operators.conditional.AndExpression;
import net.sf.jsqlparser.schema.Table;
import org.springframework.stereotype.Component;

import java.util.List;

import static cn.iocoder.foodnexus.framework.security.core.util.SecurityFrameworkUtils.skipPermissionCheck;

/**
 * 基于 {@link CustomerPermissionRule} 的数据权限处理器
 *
 * 它的底层，是基于 MyBatis Plus 的 <a href="https://baomidou.com/plugins/data-permission/">数据权限插件</a>
 * 核心原理：它会在 SQL 执行前拦截 SQL 语句，并根据用户权限动态添加权限相关的 SQL 片段。这样，只有用户有权限访问的数据才会被查询出来
 *
 * @author 芋道源码
 */
@RequiredArgsConstructor
@Component
public class CustomerPermissionRuleHandler implements MultiDataPermissionHandler {

    private final CustomerPermissionRuleFactory ruleFactory;

    public static final String TABLE_NAME = "product_spu";

    @Override
    public Expression getSqlSegment(Table table, Expression where, String mappedStatementId) {
        // 特殊：跨租户访问
        if (skipPermissionCheck()) {
            return null;
        }

        // 获得 Mapper 对应的数据权限的规则
        List<CustomerPermissionRule> rules = ruleFactory.getCustomerPermissionRule(mappedStatementId);
        if (CollUtil.isEmpty(rules)) {
            return null;
        }

        // 生成条件
        Expression allExpression = null;
        for (CustomerPermissionRule rule : rules) {
            // 判断表名是否匹配
            String tableName = MyBatisUtils.getTableName(table);
            if (!TABLE_NAME.equalsIgnoreCase(tableName)) {
                continue;
            }

            // 单条规则的条件
            Expression oneExpress = rule.getExpression(tableName, table.getAlias());
            if (oneExpress == null) {
                continue;
            }
            // 拼接到 allExpression 中
            allExpression = allExpression == null ? oneExpress
                    : new AndExpression(allExpression, oneExpress);
        }
        return allExpression;
    }

}
