package cn.iocoder.foodnexus.module.operations.controller.admin.customerrequire;

import org.springframework.web.bind.annotation.*;
import jakarta.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import jakarta.validation.constraints.*;
import jakarta.validation.*;
import jakarta.servlet.http.*;
import java.util.*;
import java.io.IOException;

import cn.iocoder.foodnexus.framework.common.pojo.PageParam;
import cn.iocoder.foodnexus.framework.common.pojo.PageResult;
import cn.iocoder.foodnexus.framework.common.pojo.CommonResult;
import cn.iocoder.foodnexus.framework.common.util.object.BeanUtils;
import static cn.iocoder.foodnexus.framework.common.pojo.CommonResult.success;

import cn.iocoder.foodnexus.framework.excel.core.util.ExcelUtils;

import cn.iocoder.foodnexus.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.foodnexus.framework.apilog.core.enums.OperateTypeEnum.*;

import cn.iocoder.foodnexus.module.operations.controller.admin.customerrequire.vo.*;
import cn.iocoder.foodnexus.module.operations.dal.dataobject.customerrequire.CustomerRequireDO;
import cn.iocoder.foodnexus.module.operations.service.customerrequire.CustomerRequireService;

@Tag(name = "管理后台 - 客户需求")
@RestController
@RequestMapping("/oper/customer-require")
@Validated
public class CustomerRequireController {

    @Resource
    private CustomerRequireService customerRequireService;


    @PostMapping("/require")
    @Operation(summary = "受理客户需求")
    @Parameter(name = "id", description = "编号", required = true)
    public CommonResult<Boolean> require(@RequestParam("id") Long id) {
        customerRequireService.require(id);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除客户需求")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('oper:customer-require:delete')")
    public CommonResult<Boolean> deleteCustomerRequire(@RequestParam("id") Long id) {
        customerRequireService.deleteCustomerRequire(id);
        return success(true);
    }

    @DeleteMapping("/delete-list")
    @Parameter(name = "ids", description = "编号", required = true)
    @Operation(summary = "批量删除客户需求")
                @PreAuthorize("@ss.hasPermission('oper:customer-require:delete')")
    public CommonResult<Boolean> deleteCustomerRequireList(@RequestParam("ids") List<Long> ids) {
        customerRequireService.deleteCustomerRequireListByIds(ids);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得客户需求")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('oper:customer-require:query')")
    public CommonResult<CustomerRequireRespVO> getCustomerRequire(@RequestParam("id") Long id) {
        CustomerRequireDO customerRequire = customerRequireService.getCustomerRequire(id);
        return success(BeanUtils.toBean(customerRequire, CustomerRequireRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得客户需求分页")
    @PreAuthorize("@ss.hasPermission('oper:customer-require:query')")
    public CommonResult<PageResult<CustomerRequireRespVO>> getCustomerRequirePage(@Valid CustomerRequirePageReqVO pageReqVO) {
        PageResult<CustomerRequireDO> pageResult = customerRequireService.getCustomerRequirePage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, CustomerRequireRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出客户需求 Excel")
    @PreAuthorize("@ss.hasPermission('oper:customer-require:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportCustomerRequireExcel(@Valid CustomerRequirePageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<CustomerRequireDO> list = customerRequireService.getCustomerRequirePage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "客户需求.xls", "数据", CustomerRequireRespVO.class,
                        BeanUtils.toBean(list, CustomerRequireRespVO.class));
    }

}