package cn.iocoder.foodnexus.framework.common.util.json.databind;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import java.io.IOException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
/**
 * @author : yanghao
 * create at:  2025/11/4  10:29
 * @description:
 */
public class FlexibleLocalDateDeserializer extends StdDeserializer<LocalDate> {

    // 定义日期字符串格式（与前端保持一致）
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    // 时区（默认北京时间）
    private static final ZoneId ZONE_ID = ZoneId.of("GMT+8");

    public FlexibleLocalDateDeserializer() {
        super(LocalDate.class);
    }

    @Override
    public LocalDate deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
        String value = p.getValueAsString();
        if (value == null || value.trim().isEmpty()) {
            return null; // 允许空值
        }

        // 尝试解析时间戳（Long类型）
        try {
            long timestamp = Long.parseLong(value.trim());
            // 时间戳转LocalDate（先转Instant，再转指定时区的LocalDate）
            return Instant.ofEpochMilli(timestamp)
                    .atZone(ZONE_ID)
                    .toLocalDate();
        } catch (NumberFormatException e) {
            // 时间戳解析失败，尝试解析日期字符串
            try {
                return LocalDate.parse(value.trim(), DATE_FORMATTER);
            } catch (DateTimeParseException ex) {
                // 两种格式都失败，抛出异常（包含具体错误信息）
                throw new IOException("无法解析LocalDate：" + value + "，支持格式：时间戳（毫秒）或 yyyy-MM-dd", ex);
            }
        }
    }

}
