package cn.iocoder.foodnexus.module.erp.service.product;

import cn.hutool.core.collection.CollUtil;
import cn.iocoder.foodnexus.framework.common.util.CommonUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import org.apache.commons.compress.utils.Lists;
import org.springframework.stereotype.Service;
import jakarta.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

import cn.iocoder.foodnexus.module.erp.controller.admin.product.vo.*;
import cn.iocoder.foodnexus.module.erp.dal.dataobject.product.ProductSupplierDO;
import cn.iocoder.foodnexus.framework.common.pojo.PageResult;
import cn.iocoder.foodnexus.framework.common.pojo.PageParam;
import cn.iocoder.foodnexus.framework.common.util.object.BeanUtils;

import cn.iocoder.foodnexus.module.erp.dal.mysql.product.ProductSupplierMapper;

import static cn.iocoder.foodnexus.framework.common.exception.util.ServiceExceptionUtil.exception;
import static cn.iocoder.foodnexus.framework.common.util.collection.CollectionUtils.convertList;
import static cn.iocoder.foodnexus.framework.common.util.collection.CollectionUtils.diffList;
import static cn.iocoder.foodnexus.module.erp.enums.ErrorCodeConstants.*;

/**
 * 供应商关联商品 Service 实现类
 *
 * @author 超级管理员
 */
@Service
@Validated
public class ProductSupplierServiceImpl implements ProductSupplierService {

    @Resource
    private ProductSupplierMapper productSupplierMapper;

    @Override
    public Long createProductSupplier(ProductSupplierSaveReqVO createReqVO) {
        // 插入
        ProductSupplierDO productSupplier = BeanUtils.toBean(createReqVO, ProductSupplierDO.class);
        productSupplierMapper.insert(productSupplier);

        // 返回
        return productSupplier.getId();
    }

    @Override
    public void updateProductSupplier(ProductSupplierSaveReqVO updateReqVO) {
        // 校验存在
        validateProductSupplierExists(updateReqVO.getId());
        // 更新
        ProductSupplierDO updateObj = BeanUtils.toBean(updateReqVO, ProductSupplierDO.class);
        productSupplierMapper.updateById(updateObj);
    }

    @Override
    public void deleteProductSupplier(Long id) {
        // 校验存在
        validateProductSupplierExists(id);
        // 删除
        productSupplierMapper.deleteById(id);
    }

    @Override
    public void deleteProductSupplierListByIds(List<Long> ids) {
    // 删除
    productSupplierMapper.deleteByIds(ids);
    }


    private void validateProductSupplierExists(Long id) {
        if (productSupplierMapper.selectById(id) == null) {
            throw exception(PRODUCT_SUPPLIER_NOT_EXISTS);
        }
    }

    @Override
    public ProductSupplierDO getProductSupplier(Long id) {
        return productSupplierMapper.selectById(id);
    }

    @Override
    public PageResult<ProductSupplierDO> getProductSupplierPage(ProductSupplierPageReqVO pageReqVO) {
        return productSupplierMapper.selectPage(pageReqVO);
    }

    @Override
    public List<Long> queryBySupplierId(Long supplierId) {
        return Optional.ofNullable(productSupplierMapper.selectList(ProductSupplierDO::getSupplierId, supplierId))
                .orElseGet(Lists::newArrayList)
                .stream()
                .map(ProductSupplierDO::getProductId)
                .collect(Collectors.toList());
    }

    /**
     * @param productId
     * @return supplierId集合
     */
    @Override
    public List<Long> queryByProductId(Long productId) {
        return Optional.ofNullable(productSupplierMapper.selectList(ProductSupplierDO::getProductId, productId))
                .orElseGet(Lists::newArrayList)
                .stream()
                .map(ProductSupplierDO::getSupplierId)
                .collect(Collectors.toList());
    }

    @Override
    public Map<Long, List<ProductSupplierDO>> groupBySupplierId(Collection<Long> productIds) {
        if (CommonUtil.isEmpty(productIds)) {
            return new HashMap<>();
        }
        List<ProductSupplierDO> productSupplierDOS = productSupplierMapper.selectList(Wrappers.<ProductSupplierDO>lambdaQuery()
                .in(ProductSupplierDO::getProductId, productIds));
        return CommonUtil.listConvertListMap(productSupplierDOS, ProductSupplierDO::getSupplierId);
    }

    @Override
    public Map<Long, Long> getCountMap(Collection<Long> productIds) {
        if (CommonUtil.isEmpty(productIds)) {
            return new HashMap<>();
        }
        Map<Long, Long> result = new HashMap<>();
        productIds.forEach(productId -> {
            result.put(productId, getCountByProductId(productId));
        });
        return result;
    }

    @Override
    public Long getCountByProductId(Long productId) {
        return productSupplierMapper.selectCount(ProductSupplierDO::getProductId, productId);
    }

}