package cn.iocoder.foodnexus.module.order.controller.admin.checktask;

import cn.iocoder.foodnexus.module.order.service.checktaskitems.CheckTaskItemsService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import jakarta.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import jakarta.validation.constraints.*;
import jakarta.validation.*;
import jakarta.servlet.http.*;
import java.util.*;
import java.io.IOException;

import cn.iocoder.foodnexus.framework.common.pojo.PageParam;
import cn.iocoder.foodnexus.framework.common.pojo.PageResult;
import cn.iocoder.foodnexus.framework.common.pojo.CommonResult;
import cn.iocoder.foodnexus.framework.common.util.object.BeanUtils;
import static cn.iocoder.foodnexus.framework.common.pojo.CommonResult.success;

import cn.iocoder.foodnexus.framework.excel.core.util.ExcelUtils;

import cn.iocoder.foodnexus.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.foodnexus.framework.apilog.core.enums.OperateTypeEnum.*;

import cn.iocoder.foodnexus.module.order.controller.admin.checktask.vo.*;
import cn.iocoder.foodnexus.module.order.dal.dataobject.checktask.CheckTaskDO;
import cn.iocoder.foodnexus.module.order.service.checktask.CheckTaskService;

@Tag(name = "管理后台 - 来料质检")
@RestController
@RequestMapping("/order/check-task")
@Validated
public class CheckTaskController {

    @Resource
    private CheckTaskService checkTaskService;

    @Autowired
    private CheckTaskItemsService checkTaskItemsService;

    @PostMapping("/create")
    @Operation(summary = "创建来料质检")
    @PreAuthorize("@ss.hasPermission('order:check-task:create')")
    public CommonResult<Long> createCheckTask(@Valid @RequestBody CheckTaskSaveReqVO createReqVO) {
        return success(checkTaskService.createCheckTask(createReqVO));
    }

    @PutMapping("/check")
    @Operation(summary = "质检")
    @PreAuthorize("@ss.hasPermission('order:check-task:update')")
    public CommonResult<Boolean> check(@Valid @RequestBody CheckTaskCheckReqVO updateReqVO) {
        checkTaskService.check(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除来料质检")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('order:check-task:delete')")
    public CommonResult<Boolean> deleteCheckTask(@RequestParam("id") Long id) {
        checkTaskService.deleteCheckTask(id);
        return success(true);
    }

    @DeleteMapping("/delete-list")
    @Parameter(name = "ids", description = "编号", required = true)
    @Operation(summary = "批量删除来料质检")
                @PreAuthorize("@ss.hasPermission('order:check-task:delete')")
    public CommonResult<Boolean> deleteCheckTaskList(@RequestParam("ids") List<Long> ids) {
        checkTaskService.deleteCheckTaskListByIds(ids);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得来料质检")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('order:check-task:query')")
    public CommonResult<CheckTaskRespVO> getCheckTask(@RequestParam("id") Long id) {
        CheckTaskDO checkTask = checkTaskService.getCheckTask(id);
        return success(BeanUtils.toBean(checkTask, CheckTaskRespVO.class, item -> {
            item.setItems(BeanUtils.toBean(checkTaskItemsService.queryByCheckId(item.getId()), CheckTaskItemsRespVO.class));
        }));
    }

    @GetMapping("/page")
    @Operation(summary = "获得来料质检分页")
    @PreAuthorize("@ss.hasPermission('order:check-task:query')")
    public CommonResult<PageResult<CheckTaskRespVO>> getCheckTaskPage(@Valid CheckTaskPageReqVO pageReqVO) {
        PageResult<CheckTaskDO> pageResult = checkTaskService.getCheckTaskPage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, CheckTaskRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出来料质检 Excel")
    @PreAuthorize("@ss.hasPermission('order:check-task:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportCheckTaskExcel(@Valid CheckTaskPageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<CheckTaskDO> list = checkTaskService.getCheckTaskPage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "来料质检.xls", "数据", CheckTaskRespVO.class,
                        BeanUtils.toBean(list, CheckTaskRespVO.class));
    }

}