package cn.iocoder.foodnexus.module.order.controller.app.customerOrder;

import cn.iocoder.foodnexus.framework.apilog.core.annotation.ApiAccessLog;
import cn.iocoder.foodnexus.framework.common.enums.UserSystemEnum;
import cn.iocoder.foodnexus.framework.common.pojo.CommonResult;
import cn.iocoder.foodnexus.framework.common.pojo.PageParam;
import cn.iocoder.foodnexus.framework.common.pojo.PageResult;
import cn.iocoder.foodnexus.framework.common.util.CommonUtil;
import cn.iocoder.foodnexus.framework.common.util.object.BeanUtils;
import cn.iocoder.foodnexus.framework.excel.core.util.ExcelUtils;
import cn.iocoder.foodnexus.module.order.controller.admin.customerorder.vo.CustomerOrderPageReqVO;
import cn.iocoder.foodnexus.module.order.controller.admin.customerorder.vo.CustomerOrderRespVO;
import cn.iocoder.foodnexus.module.order.controller.admin.customerorder.vo.CustomerOrderSaveReqVO;
import cn.iocoder.foodnexus.module.order.controller.app.customerOrder.vo.AppCustomerOrderRemarkReqVO;
import cn.iocoder.foodnexus.module.order.controller.app.customerOrder.vo.AppCustomerOrderSaveReqVO;
import cn.iocoder.foodnexus.module.order.controller.app.customerOrder.vo.AppCustomerOrderSaveReqVO;
import cn.iocoder.foodnexus.module.order.dal.dataobject.customerorder.CustomerOrderDO;
import cn.iocoder.foodnexus.module.order.service.customerorder.CustomerOrderService;
import cn.iocoder.foodnexus.module.system.annotations.AppSystemAuth;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.Valid;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.sql.rowset.serial.SerialException;
import java.io.IOException;
import java.util.List;

import static cn.iocoder.foodnexus.framework.apilog.core.enums.OperateTypeEnum.EXPORT;
import static cn.iocoder.foodnexus.framework.common.pojo.CommonResult.success;

@Tag(name = "APP - 客户总订单")
@RestController
@RequestMapping("/order/customer-order")
@Validated
@AppSystemAuth(UserSystemEnum.CUSTOMER)
public class AppCustomerOrderController {

    @Resource
    private CustomerOrderService customerOrderService;

    @PostMapping("/create")
    @Operation(summary = "创建客户总订单")
    public CommonResult<Long> createCustomerOrder(@Valid @RequestBody AppCustomerOrderSaveReqVO createReqVO) {
        return success(customerOrderService.appCreateCustomerOrder(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新客户总订单")
    public CommonResult<Boolean> updateCustomerOrder(@Valid @RequestBody AppCustomerOrderSaveReqVO updateReqVO) throws SerialException {
        if (CommonUtil.isEmpty(updateReqVO.getId())) {
            throw new SerialException("订单id不能为空");
        }
        customerOrderService.appUpdateCustomerOrder(updateReqVO);
        return success(true);
    }

    @PostMapping("cancel")
    @Operation(summary = "取消客户订单")
    public CommonResult<Boolean> cancel(@Valid @RequestBody AppCustomerOrderRemarkReqVO reqVO) {
        customerOrderService.cancel(reqVO);
        return success(Boolean.TRUE);
    }

    @PostMapping("receipt")
    @Operation(summary = "签收客户订单")
    public CommonResult<Boolean> receipt(@Valid @RequestBody AppCustomerOrderRemarkReqVO reqVO) {
        customerOrderService.receipt(reqVO);
        return success(Boolean.TRUE);
    }

    @DeleteMapping("/delete-list")
    @Parameter(name = "ids", description = "编号", required = true)
    @Operation(summary = "批量删除客户总订单")
    public CommonResult<Boolean> deleteCustomerOrderList(@RequestParam("ids") List<Long> ids) {
        customerOrderService.deleteCustomerOrderListByIds(ids);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得客户总订单")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    public CommonResult<CustomerOrderRespVO> getCustomerOrder(@RequestParam("id") Long id) {
        CustomerOrderDO customerOrder = customerOrderService.getCustomerOrder(id);
        return success(BeanUtils.toBean(customerOrder, CustomerOrderRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得客户总订单分页")
    public CommonResult<PageResult<CustomerOrderRespVO>> getCustomerOrderPage(@Valid CustomerOrderPageReqVO pageReqVO) {
        PageResult<CustomerOrderDO> pageResult = customerOrderService.getCustomerOrderPage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, CustomerOrderRespVO.class));
    }

}