package cn.iocoder.foodnexus.module.operations.controller.admin.deliverystaff;

import cn.iocoder.foodnexus.module.operations.service.deliverystaffcustomer.DeliveryStaffCustomerService;
import cn.iocoder.foodnexus.module.operations.service.vehicleinfo.VehicleInfoService;
import cn.iocoder.foodnexus.module.system.dal.dataobject.user.AdminUserDO;
import cn.iocoder.foodnexus.module.system.service.user.AdminUserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import jakarta.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import jakarta.validation.constraints.*;
import jakarta.validation.*;
import jakarta.servlet.http.*;
import java.util.*;
import java.io.IOException;

import cn.iocoder.foodnexus.framework.common.pojo.PageParam;
import cn.iocoder.foodnexus.framework.common.pojo.PageResult;
import cn.iocoder.foodnexus.framework.common.pojo.CommonResult;
import cn.iocoder.foodnexus.framework.common.util.object.BeanUtils;
import static cn.iocoder.foodnexus.framework.common.pojo.CommonResult.success;

import cn.iocoder.foodnexus.framework.excel.core.util.ExcelUtils;

import cn.iocoder.foodnexus.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.foodnexus.framework.apilog.core.enums.OperateTypeEnum.*;

import cn.iocoder.foodnexus.module.operations.controller.admin.deliverystaff.vo.*;
import cn.iocoder.foodnexus.module.operations.dal.dataobject.deliverystaff.DeliveryStaffDO;
import cn.iocoder.foodnexus.module.operations.service.deliverystaff.DeliveryStaffService;

@Tag(name = "管理后台 - 配送员信息")
@RestController
@RequestMapping("/operation/delivery-staff")
@Validated
public class DeliveryStaffController {

    @Resource
    private DeliveryStaffService deliveryStaffService;

    @Resource
    private DeliveryStaffCustomerService deliveryStaffCustomerService;

    @Resource
    private VehicleInfoService vehicleInfoService;

    @Autowired
    private AdminUserService userService;

    @PostMapping("/create")
    @Operation(summary = "创建配送员信息")
    @PreAuthorize("@ss.hasPermission('operation:delivery-staff:create')")
    public CommonResult<Long> createDeliveryStaff(@Valid @RequestBody DeliveryStaffSaveReqVO createReqVO) {
        return success(deliveryStaffService.createDeliveryStaff(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新配送员信息")
    @PreAuthorize("@ss.hasPermission('operation:delivery-staff:update')")
    public CommonResult<Boolean> updateDeliveryStaff(@Valid @RequestBody DeliveryStaffSaveReqVO updateReqVO) {
        deliveryStaffService.updateDeliveryStaff(updateReqVO);
        return success(true);
    }

    @PutMapping("/bind-customer")
    @Operation(summary = "配送员绑定客户")
    @PreAuthorize("@ss.hasPermission('operation:delivery-staff:update')")
    public CommonResult<Boolean> bindCustomer(@Valid @RequestBody DeliveryStaffBindCustomerReqVO updateReqVO) {
        deliveryStaffService.bindCustomer(updateReqVO);
        return success(true);
    }

    @PutMapping("bind-customer-release")
    @Operation(summary = "配送员解绑客户")
    @PreAuthorize("@ss.hasPermission('operation:delivery-staff:update')")
    public CommonResult<Boolean> bindCustomerRelease(@Valid @RequestBody DeliveryStaffBindCustomerReqVO updateReqVO) {
        deliveryStaffService.bindCustomerRelease(updateReqVO);
        return success(true);
    }

    @PutMapping("/bind-dept")
    @Operation(summary = "配送员绑定灶点")
    @PreAuthorize("@ss.hasPermission('operation:delivery-staff:update')")
    public CommonResult<Boolean> bindDept(@Valid @RequestBody DeliveryStaffBindDeptReqVO updateReqVO) {
        deliveryStaffService.bindDept(updateReqVO);
        return success(true);
    }

    @PutMapping("bind-dept-release")
    @Operation(summary = "配送员解绑灶点")
    @PreAuthorize("@ss.hasPermission('operation:delivery-staff:update')")
    public CommonResult<Boolean> bindDeptRelease(@Valid @RequestBody DeliveryStaffBindDeptReqVO updateReqVO) {
        deliveryStaffService.bindDeptRelease(updateReqVO);
        return success(true);
    }


    @DeleteMapping("/delete")
    @Operation(summary = "删除配送员信息")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('operation:delivery-staff:delete')")
    public CommonResult<Boolean> deleteDeliveryStaff(@RequestParam("id") Long id) {
        deliveryStaffService.deleteDeliveryStaff(id);
        return success(true);
    }

    @DeleteMapping("/delete-list")
    @Parameter(name = "ids", description = "编号", required = true)
    @Operation(summary = "批量删除配送员信息")
                @PreAuthorize("@ss.hasPermission('operation:delivery-staff:delete')")
    public CommonResult<Boolean> deleteDeliveryStaffList(@RequestParam("ids") List<Long> ids) {
        deliveryStaffService.deleteDeliveryStaffListByIds(ids);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得配送员信息")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('operation:delivery-staff:query')")
    public CommonResult<DeliveryStaffRespVO> getDeliveryStaff(@RequestParam("id") Long id) {
        DeliveryStaffDO deliveryStaff = deliveryStaffService.getDeliveryStaff(id);
        return success(BeanUtils.toBean(deliveryStaff, DeliveryStaffRespVO.class, item -> {
            item.setCustomerList(deliveryStaffCustomerService.queryCustomerListByDeliveryStaffId(item.getId()));
            item.setVehicleInfo(vehicleInfoService.queryByStaffId(item.getId()));
            item.setCustomerDeptList(deliveryStaffCustomerService.queryDeptListByStaffId(item.getId()));
            item.setUserName(Optional.ofNullable(userService.getUser(item.getUserId())).map(AdminUserDO::getUsername).orElse(""));
        }));
    }

    @GetMapping("/page")
    @Operation(summary = "获得配送员信息分页")
    @PreAuthorize("@ss.hasPermission('operation:delivery-staff:query')")
    public CommonResult<PageResult<DeliveryStaffRespVO>> getDeliveryStaffPage(@Valid DeliveryStaffPageReqVO pageReqVO) {
        PageResult<DeliveryStaffDO> pageResult = deliveryStaffService.getDeliveryStaffPage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, DeliveryStaffRespVO.class, item -> {
            item.setUserName(Optional.ofNullable(userService.getUser(item.getUserId())).map(AdminUserDO::getUsername).orElse(""));
        }));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出配送员信息 Excel")
    @PreAuthorize("@ss.hasPermission('operation:delivery-staff:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportDeliveryStaffExcel(@Valid DeliveryStaffPageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<DeliveryStaffDO> list = deliveryStaffService.getDeliveryStaffPage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "配送员信息.xls", "数据", DeliveryStaffRespVO.class,
                        BeanUtils.toBean(list, DeliveryStaffRespVO.class));
    }

}