package cn.iocoder.foodnexus.module.operations.controller.admin.scoringweight;

import org.springframework.web.bind.annotation.*;
import jakarta.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import jakarta.validation.constraints.*;
import jakarta.validation.*;
import jakarta.servlet.http.*;
import java.util.*;
import java.io.IOException;

import cn.iocoder.foodnexus.framework.common.pojo.PageParam;
import cn.iocoder.foodnexus.framework.common.pojo.PageResult;
import cn.iocoder.foodnexus.framework.common.pojo.CommonResult;
import cn.iocoder.foodnexus.framework.common.util.object.BeanUtils;
import static cn.iocoder.foodnexus.framework.common.pojo.CommonResult.success;

import cn.iocoder.foodnexus.framework.excel.core.util.ExcelUtils;

import cn.iocoder.foodnexus.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.foodnexus.framework.apilog.core.enums.OperateTypeEnum.*;

import cn.iocoder.foodnexus.module.operations.controller.admin.scoringweight.vo.*;
import cn.iocoder.foodnexus.module.operations.dal.dataobject.scoringweight.ScoringWeightDO;
import cn.iocoder.foodnexus.module.operations.service.scoringweight.ScoringWeightService;

@Tag(name = "管理后台 - 评分权重")
@RestController
@RequestMapping("/operations/scoring-weight")
@Validated
public class ScoringWeightController {

    @Resource
    private ScoringWeightService scoringWeightService;

    @PostMapping("/create")
    @Operation(summary = "创建评分权重")
    @PreAuthorize("@ss.hasPermission('operations:scoring-weight:create')")
    public CommonResult<Long> createScoringWeight(@Valid @RequestBody ScoringWeightSaveReqVO createReqVO) {
        return success(scoringWeightService.createScoringWeight(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新评分权重")
    @PreAuthorize("@ss.hasPermission('operations:scoring-weight:update')")
    public CommonResult<Boolean> updateScoringWeight(@Valid @RequestBody ScoringWeightUpateReqVO updateReqVO) {
        scoringWeightService.updateScoringWeight(updateReqVO);
        return success(true);
    }

    @PostMapping("/edit")
    @Operation(summary = "编辑权重")
    public CommonResult<Boolean> edit(@Valid @RequestBody ScoringWeightEditReqVO reqVO) {
        scoringWeightService.edit(reqVO);
        return success(Boolean.TRUE);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除评分权重")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('operations:scoring-weight:delete')")
    public CommonResult<Boolean> deleteScoringWeight(@RequestParam("id") Long id) {
        scoringWeightService.deleteScoringWeight(id);
        return success(true);
    }

    @DeleteMapping("/delete-list")
    @Parameter(name = "ids", description = "编号", required = true)
    @Operation(summary = "批量删除评分权重")
                @PreAuthorize("@ss.hasPermission('operations:scoring-weight:delete')")
    public CommonResult<Boolean> deleteScoringWeightList(@RequestParam("ids") List<Long> ids) {
        scoringWeightService.deleteScoringWeightListByIds(ids);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得评分权重")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('operations:scoring-weight:query')")
    public CommonResult<ScoringWeightRespVO> getScoringWeight(@RequestParam("id") Long id) {
        ScoringWeightDO scoringWeight = scoringWeightService.getScoringWeight(id);
        return success(BeanUtils.toBean(scoringWeight, ScoringWeightRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得评分权重分页")
    @PreAuthorize("@ss.hasPermission('operations:scoring-weight:query')")
    public CommonResult<PageResult<ScoringWeightRespVO>> getScoringWeightPage(@Valid ScoringWeightPageReqVO pageReqVO) {
        PageResult<ScoringWeightDO> pageResult = scoringWeightService.getScoringWeightPage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, ScoringWeightRespVO.class));
    }

    @GetMapping("/list")
    @Operation(summary = "获取全部评分权重")
    public CommonResult<List<ScoringWeightRespVO>> getScoringWeightList() {
        return success(BeanUtils.toBean(scoringWeightService.listAll(), ScoringWeightRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出评分权重 Excel")
    @PreAuthorize("@ss.hasPermission('operations:scoring-weight:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportScoringWeightExcel(@Valid ScoringWeightPageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<ScoringWeightDO> list = scoringWeightService.getScoringWeightPage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "评分权重.xls", "数据", ScoringWeightRespVO.class,
                        BeanUtils.toBean(list, ScoringWeightRespVO.class));
    }

}