package cn.iocoder.foodnexus.module.operations.controller.admin.vehicleinfo;

import org.springframework.web.bind.annotation.*;
import jakarta.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import jakarta.validation.constraints.*;
import jakarta.validation.*;
import jakarta.servlet.http.*;
import java.util.*;
import java.io.IOException;

import cn.iocoder.foodnexus.framework.common.pojo.PageParam;
import cn.iocoder.foodnexus.framework.common.pojo.PageResult;
import cn.iocoder.foodnexus.framework.common.pojo.CommonResult;
import cn.iocoder.foodnexus.framework.common.util.object.BeanUtils;
import static cn.iocoder.foodnexus.framework.common.pojo.CommonResult.success;

import cn.iocoder.foodnexus.framework.excel.core.util.ExcelUtils;

import cn.iocoder.foodnexus.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.foodnexus.framework.apilog.core.enums.OperateTypeEnum.*;

import cn.iocoder.foodnexus.module.operations.controller.admin.vehicleinfo.vo.*;
import cn.iocoder.foodnexus.module.operations.dal.dataobject.vehicleinfo.VehicleInfoDO;
import cn.iocoder.foodnexus.module.operations.service.vehicleinfo.VehicleInfoService;

@Tag(name = "管理后台 - 车辆信息")
@RestController
@RequestMapping("/operations/vehicle-info")
@Validated
public class VehicleInfoController {

    @Resource
    private VehicleInfoService vehicleInfoService;

    @PostMapping("/create")
    @Operation(summary = "创建车辆信息")
    @PreAuthorize("@ss.hasPermission('operations:vehicle-info:create')")
    public CommonResult<Long> createVehicleInfo(@Valid @RequestBody VehicleInfoSaveReqVO createReqVO) {
        return success(vehicleInfoService.createVehicleInfo(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新车辆信息")
    @PreAuthorize("@ss.hasPermission('operations:vehicle-info:update')")
    public CommonResult<Boolean> updateVehicleInfo(@Valid @RequestBody VehicleInfoSaveReqVO updateReqVO) {
        vehicleInfoService.updateVehicleInfo(updateReqVO);
        return success(true);
    }

    @PutMapping("/bind")
    @Operation(summary = "人车绑定")
    @PreAuthorize("@ss.hasPermission('operations:vehicle-info:update')")
    public CommonResult<Boolean> vehicleBind(@Valid @RequestBody VehicleInfoBindReqVO updateReqVO) {
        vehicleInfoService.vehicleBind(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/release-bind")
    @Operation(summary = "解除人车绑定")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('operations:vehicle-info:update')")
    public CommonResult<Boolean> releaseBind(@RequestParam("id") Long id) {
        vehicleInfoService.releaseBind(id);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除车辆信息")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('operations:vehicle-info:delete')")
    public CommonResult<Boolean> deleteVehicleInfo(@RequestParam("id") Long id) {
        vehicleInfoService.deleteVehicleInfo(id);
        return success(true);
    }

    @DeleteMapping("/delete-list")
    @Parameter(name = "ids", description = "编号", required = true)
    @Operation(summary = "批量删除车辆信息")
                @PreAuthorize("@ss.hasPermission('operations:vehicle-info:delete')")
    public CommonResult<Boolean> deleteVehicleInfoList(@RequestParam("ids") List<Long> ids) {
        vehicleInfoService.deleteVehicleInfoListByIds(ids);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得车辆信息")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('operations:vehicle-info:query')")
    public CommonResult<VehicleInfoRespVO> getVehicleInfo(@RequestParam("id") Long id) {
        VehicleInfoDO vehicleInfo = vehicleInfoService.getVehicleInfo(id);
        return success(BeanUtils.toBean(vehicleInfo, VehicleInfoRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得车辆信息分页")
    @PreAuthorize("@ss.hasPermission('operations:vehicle-info:query')")
    public CommonResult<PageResult<VehicleInfoRespVO>> getVehicleInfoPage(@Valid VehicleInfoPageReqVO pageReqVO) {
        PageResult<VehicleInfoDO> pageResult = vehicleInfoService.getVehicleInfoPage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, VehicleInfoRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出车辆信息 Excel")
    @PreAuthorize("@ss.hasPermission('operations:vehicle-info:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportVehicleInfoExcel(@Valid VehicleInfoPageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<VehicleInfoDO> list = vehicleInfoService.getVehicleInfoPage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "车辆信息.xls", "数据", VehicleInfoRespVO.class,
                        BeanUtils.toBean(list, VehicleInfoRespVO.class));
    }

}