package cn.iocoder.foodnexus.module.erp.controller.admin.sale;

import cn.hutool.core.collection.CollUtil;
import cn.iocoder.foodnexus.framework.apilog.core.annotation.ApiAccessLog;
import cn.iocoder.foodnexus.framework.common.pojo.CommonResult;
import cn.iocoder.foodnexus.framework.common.pojo.PageParam;
import cn.iocoder.foodnexus.framework.common.pojo.PageResult;
import cn.iocoder.foodnexus.framework.common.util.CommonUtil;
import cn.iocoder.foodnexus.framework.common.util.collection.MapUtils;
import cn.iocoder.foodnexus.framework.common.util.object.BeanUtils;
import cn.iocoder.foodnexus.framework.excel.core.util.ExcelUtils;
import cn.iocoder.foodnexus.module.erp.api.service.ErpSupplierApi;
import cn.iocoder.foodnexus.module.erp.controller.admin.sale.vo.order.ErpSaleOrderPageReqVO;
import cn.iocoder.foodnexus.module.erp.controller.admin.sale.vo.order.ErpSaleOrderRespVO;
import cn.iocoder.foodnexus.module.erp.dal.dataobject.purchase.ErpPurchaseOrderDO;
import cn.iocoder.foodnexus.module.erp.dal.dataobject.sale.ErpCustomerDO;
import cn.iocoder.foodnexus.module.erp.dal.dataobject.sale.ErpSaleOrderDO;
import cn.iocoder.foodnexus.module.erp.dal.dataobject.sale.ErpSaleOrderItemDO;
import cn.iocoder.foodnexus.module.erp.enums.SaleOrderPickUpStatus;
import cn.iocoder.foodnexus.module.erp.service.purchase.ErpPurchaseOrderService;
import cn.iocoder.foodnexus.module.erp.service.sale.ErpCustomerService;
import cn.iocoder.foodnexus.module.erp.service.sale.ErpSaleOrderService;
import cn.iocoder.foodnexus.module.erp.service.stock.ErpStockService;
import cn.iocoder.foodnexus.module.order.api.CustomerOrderApi;
import cn.iocoder.foodnexus.module.order.api.DeliveryStaffApi;
import cn.iocoder.foodnexus.module.order.dto.CustomerOrderDTO;
import cn.iocoder.foodnexus.module.order.dto.CustomerOrderItemDTO;
import cn.iocoder.foodnexus.module.order.dto.DeliveryStaffSimpleInfo;
import cn.iocoder.foodnexus.module.product.api.dto.ProductInfo;
import cn.iocoder.foodnexus.module.product.service.spu.ProductSpuService;
import cn.iocoder.foodnexus.module.system.api.user.AdminUserApi;
import cn.iocoder.foodnexus.module.system.api.user.dto.AdminUserRespDTO;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.Valid;
import org.apache.commons.compress.utils.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static cn.iocoder.foodnexus.framework.apilog.core.enums.OperateTypeEnum.EXPORT;
import static cn.iocoder.foodnexus.framework.common.pojo.CommonResult.success;
import static cn.iocoder.foodnexus.framework.common.util.collection.CollectionUtils.convertMultiMap;
import static cn.iocoder.foodnexus.framework.common.util.collection.CollectionUtils.convertSet;

@Tag(name = "管理后台 - ERP 销售订单")
@RestController
@RequestMapping("/erp/sale-order")
@Validated
public class ErpSaleOrderController {

    @Resource
    private ErpSaleOrderService saleOrderService;
    @Resource
    private ErpStockService stockService;
    @Resource
    private ProductSpuService productService;
    @Resource
    private ErpCustomerService customerService;

    @Resource
    private AdminUserApi adminUserApi;

    @Autowired
    private DeliveryStaffApi deliveryStaffApi;

    @Autowired
    private CustomerOrderApi customerOrderApi;

    @Autowired
    private ErpPurchaseOrderService purchaseOrderService;

    @Autowired
    private ErpSupplierApi supplierApi;

    /*@PostMapping("/create")
    @Operation(summary = "创建销售订单")
    @PreAuthorize("@ss.hasPermission('erp:sale-out:create')")
    public CommonResult<Long> createSaleOrder(@Valid @RequestBody ErpSaleOrderSaveReqVO createReqVO) {
        return success(saleOrderService.createSaleOrder(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新销售订单")
    @PreAuthorize("@ss.hasPermission('erp:sale-out:update')")
    public CommonResult<Boolean> updateSaleOrder(@Valid @RequestBody ErpSaleOrderSaveReqVO updateReqVO) {
        saleOrderService.updateSaleOrder(updateReqVO);
        return success(true);
    }

    @PutMapping("/update-status")
    @Operation(summary = "更新销售订单的状态")
    @PreAuthorize("@ss.hasPermission('erp:sale-out:update-status')")
    public CommonResult<Boolean> updateSaleOrderStatus(@RequestParam("id") Long id,
                                                      @RequestParam("status") Integer status) {
        saleOrderService.updateSaleOrderStatus(id, status);
        return success(true);
    }*/

    @PostMapping("/delivery-staff")
    @Operation(summary = "选择配送人员")
    @PreAuthorize("@ss.hasPermission('erp:sale-out:update')")
    public CommonResult<Boolean> updateDeliveryStaff(@RequestParam("id") Long id,
                                                       @RequestParam("deliveryStaffId") Long deliveryStaffId) {
        return success(saleOrderService.updateDeliveryStaff(id, deliveryStaffId));
    }

    @PostMapping("/pick-up")
    @Operation(summary = "检获")
    @Parameter(name = "id", description = "销售订单id", required = true)
    @Parameter(name = "itemId", description = "销售订单子id", required = true)
    public CommonResult<Boolean> pickUp(@RequestParam("id") Long id,
                                        @RequestParam("itemId") Long itemId) {
        saleOrderService.updatePickUp(id, itemId);
        return success(Boolean.TRUE);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除销售订单")
    @Parameter(name = "ids", description = "编号数组", required = true)
    @PreAuthorize("@ss.hasPermission('erp:sale-out:delete')")
    public CommonResult<Boolean> deleteSaleOrder(@RequestParam("ids") List<Long> ids) {
        saleOrderService.deleteSaleOrder(ids);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得销售订单")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('erp:sale-out:query')")
    public CommonResult<ErpSaleOrderRespVO> getSaleOrder(@RequestParam("id") Long id) {
        ErpSaleOrderDO saleOrder = saleOrderService.getSaleOrder(id);
        if (saleOrder == null) {
            return success(null);
        }
        List<ErpSaleOrderItemDO> saleOrderItemList = saleOrderService.getSaleOrderItemListByOrderId(id);
        List<CustomerOrderItemDTO> customerOrderItems = customerOrderApi.queryItemsByIds(CommonUtil.listConvert(saleOrderItemList, ErpSaleOrderItemDO::getCustomerOrderItemId));
        Map<Long, CustomerOrderItemDTO> customerOrderItemMap = CommonUtil.listConvertMap(customerOrderItems, CustomerOrderItemDTO::getId);
        Map<Long, ErpPurchaseOrderDO> purchaseOrderMap = purchaseOrderService.getMap(CommonUtil.listConvert(saleOrderItemList, ErpSaleOrderItemDO::getPurchaseOrderId));
        Map<Long, String> supplierMap = supplierApi.queryNameMapByIds(CommonUtil.listConvertSet(saleOrderItemList, ErpSaleOrderItemDO::getSupplierId));
        return success(BeanUtils.toBean(saleOrder, ErpSaleOrderRespVO.class, saleOrderVO -> {
            saleOrderVO.setCustomerOrderCode(
                    Optional.ofNullable(customerOrderApi.queryById(saleOrderVO.getCustomerOrderId())).map(CustomerOrderDTO::getCode).orElse(""));
            if (CommonUtil.isNotEmpty(saleOrderVO.getDeliveryStaffId())) {
                saleOrderVO.setDeliveryStaffInfo(deliveryStaffApi.querybyStaffId(saleOrderVO.getDeliveryStaffId()));
            }
            saleOrderVO.setItems(BeanUtils.toBean(saleOrderItemList, ErpSaleOrderRespVO.Item.class, item -> {
                MapUtils.findAndThen(supplierMap, item.getSupplierId(), item::setSupplierName);
                MapUtils.findAndThen(purchaseOrderMap, item.getPurchaseOrderId(), purchaseOrder ->
                        item.setPurchaseOrderNo(purchaseOrder.getNo()));
                BigDecimal stockCount = stockService.getStockCount(item.getProductId());
                item.setStockCount(stockCount != null ? stockCount : BigDecimal.ZERO);
                MapUtils.findAndThen(customerOrderItemMap, item.getCustomerOrderItemId(), customerOrderItem ->
                        item.setProductInfo(customerOrderItem.getProductInfo()));
            }));}
        ));
    }

    @GetMapping("/page")
    @Operation(summary = "获得销售订单分页")
    @PreAuthorize("@ss.hasPermission('erp:sale-out:query')")
    public CommonResult<PageResult<ErpSaleOrderRespVO>> getSaleOrderPage(@Valid ErpSaleOrderPageReqVO pageReqVO) {
        PageResult<ErpSaleOrderDO> pageResult = saleOrderService.getSaleOrderPage(pageReqVO);
        return success(buildSaleOrderVOPageResult(pageResult));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出销售订单 Excel")
    @PreAuthorize("@ss.hasPermission('erp:sale-out:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportSaleOrderExcel(@Valid ErpSaleOrderPageReqVO pageReqVO,
                                    HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<ErpSaleOrderRespVO> list = buildSaleOrderVOPageResult(saleOrderService.getSaleOrderPage(pageReqVO)).getList();
        // 导出 Excel
        ExcelUtils.write(response, "销售订单.xls", "数据", ErpSaleOrderRespVO.class, list);
    }

    private PageResult<ErpSaleOrderRespVO> buildSaleOrderVOPageResult(PageResult<ErpSaleOrderDO> pageResult) {
        if (CollUtil.isEmpty(pageResult.getList())) {
            return PageResult.empty(pageResult.getTotal());
        }
        // 1.1 订单项
        List<ErpSaleOrderItemDO> saleOrderItemList = saleOrderService.getSaleOrderItemListByOrderIds(
                convertSet(pageResult.getList(), ErpSaleOrderDO::getId));
        Map<Long, List<ErpSaleOrderItemDO>> saleOrderItemMap = convertMultiMap(saleOrderItemList, ErpSaleOrderItemDO::getOrderId);
        // 1.3 客户信息
        Map<Long, ErpCustomerDO> customerMap = customerService.getCustomerMap(
                convertSet(pageResult.getList(), ErpSaleOrderDO::getCustomerId));
        // 1.4 管理员信息
        Map<Long, AdminUserRespDTO> userMap = adminUserApi.getUserMap(
                convertSet(pageResult.getList(), saleOrder -> Long.parseLong(saleOrder.getCreator())));

        // 配送人员
        Map<Long, DeliveryStaffSimpleInfo> staffMap = deliveryStaffApi.getMap(CommonUtil.listConvertSet(pageResult.getList(), ErpSaleOrderDO::getDeliveryStaffId));

        // 客户订单编号
        Map<Long, CustomerOrderDTO> orderMap = customerOrderApi.getOrderMap(CommonUtil.listConvert(pageResult.getList(), ErpSaleOrderDO::getCustomerOrderId));

        List<CustomerOrderItemDTO> customerOrderItems = customerOrderApi.queryItemsByIds(CommonUtil.listConvert(saleOrderItemList, ErpSaleOrderItemDO::getCustomerOrderItemId));
        Map<Long, CustomerOrderItemDTO> customerOrderItemMap = CommonUtil.listConvertMap(customerOrderItems, CustomerOrderItemDTO::getId);

        Map<Long, ErpPurchaseOrderDO> purchaseOrderMap = purchaseOrderService.getMap(CommonUtil.listConvert(saleOrderItemList, ErpSaleOrderItemDO::getPurchaseOrderId));
        Map<Long, String> supplierMap = supplierApi.queryNameMapByIds(CommonUtil.listConvertSet(saleOrderItemList, ErpSaleOrderItemDO::getSupplierId));

        final String pickUpStatusDetails = "已到%s 已拣%s 共%s";
        // 2. 开始拼接
        return BeanUtils.toBean(pageResult, ErpSaleOrderRespVO.class, saleOrder -> {
            saleOrder.setItems(BeanUtils.toBean(saleOrderItemMap.get(saleOrder.getId()), ErpSaleOrderRespVO.Item.class, item -> {
                MapUtils.findAndThen(purchaseOrderMap, item.getPurchaseOrderId(), purchaseOrder ->
                        item.setPurchaseOrderNo(purchaseOrder.getNo()));
                MapUtils.findAndThen(customerOrderItemMap, item.getCustomerOrderItemId(), customerOrderItem ->
                        item.setProductInfo(customerOrderItem.getProductInfo()));
                MapUtils.findAndThen(supplierMap, item.getSupplierId(), item::setSupplierName);
            }));
            saleOrder.setProductNames(CollUtil.join(saleOrder.getItems(), "，", item ->
                    Optional.ofNullable(item.getProductInfo()).map(ProductInfo::getName).orElse("")));
            MapUtils.findAndThen(customerMap, saleOrder.getCustomerId(), supplier -> saleOrder.setCustomerName(supplier.getName()));
            MapUtils.findAndThen(userMap, Long.parseLong(saleOrder.getCreator()), user -> saleOrder.setCreatorName(user.getNickname()));
            MapUtils.findAndThen(staffMap, saleOrder.getDeliveryStaffId(), saleOrder::setDeliveryStaffInfo);
            MapUtils.findAndThen(orderMap, saleOrder.getCustomerOrderId(), order -> saleOrder.setCustomerOrderCode(order.getCode()));
            saleOrder.setPickUpStatusDetails(String.format(pickUpStatusDetails,
                    Optional.ofNullable(saleOrder.getItems()).orElse(Lists.newArrayList()).stream().filter(item -> SaleOrderPickUpStatus.ARRIVAL.equals(item.getPickUpStatus())).count(),
                    Optional.ofNullable(saleOrder.getItems()).orElse(Lists.newArrayList()).stream().filter(item -> !SaleOrderPickUpStatus.TO_BE.equals(item.getPickUpStatus())).count(),
                    Optional.ofNullable(saleOrder.getItems()).orElse(Lists.newArrayList()).size()
            ));
        });
    }

}