package cn.iocoder.foodnexus.module.order.controller.admin.customerorder;

import org.springframework.web.bind.annotation.*;
import jakarta.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import jakarta.validation.constraints.*;
import jakarta.validation.*;
import jakarta.servlet.http.*;
import java.util.*;
import java.io.IOException;

import cn.iocoder.foodnexus.framework.common.pojo.PageParam;
import cn.iocoder.foodnexus.framework.common.pojo.PageResult;
import cn.iocoder.foodnexus.framework.common.pojo.CommonResult;
import cn.iocoder.foodnexus.framework.common.util.object.BeanUtils;
import static cn.iocoder.foodnexus.framework.common.pojo.CommonResult.success;

import cn.iocoder.foodnexus.framework.excel.core.util.ExcelUtils;

import cn.iocoder.foodnexus.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.foodnexus.framework.apilog.core.enums.OperateTypeEnum.*;

import cn.iocoder.foodnexus.module.order.controller.admin.customerorder.vo.*;
import cn.iocoder.foodnexus.module.order.dal.dataobject.customerorder.CustomerOrderDO;
import cn.iocoder.foodnexus.module.order.service.customerorder.CustomerOrderService;

@Tag(name = "管理后台 - 客户总订单")
@RestController
@RequestMapping("/order/customer-order")
@Validated
public class CustomerOrderController {

    @Resource
    private CustomerOrderService customerOrderService;

    @PostMapping("/create")
    @Operation(summary = "创建客户总订单")
    @PreAuthorize("@ss.hasPermission('order:customer-order:create')")
    public CommonResult<Long> createCustomerOrder(@Valid @RequestBody CustomerOrderSaveReqVO createReqVO) {
        return success(customerOrderService.createCustomerOrder(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新客户总订单")
    @PreAuthorize("@ss.hasPermission('order:customer-order:update')")
    public CommonResult<Boolean> updateCustomerOrder(@Valid @RequestBody CustomerOrderSaveReqVO updateReqVO) {
        customerOrderService.updateCustomerOrder(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除客户总订单")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('order:customer-order:delete')")
    public CommonResult<Boolean> deleteCustomerOrder(@RequestParam("id") Long id) {
        customerOrderService.deleteCustomerOrder(id);
        return success(true);
    }

    @DeleteMapping("/delete-list")
    @Parameter(name = "ids", description = "编号", required = true)
    @Operation(summary = "批量删除客户总订单")
                @PreAuthorize("@ss.hasPermission('order:customer-order:delete')")
    public CommonResult<Boolean> deleteCustomerOrderList(@RequestParam("ids") List<Long> ids) {
        customerOrderService.deleteCustomerOrderListByIds(ids);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得客户总订单")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('order:customer-order:query')")
    public CommonResult<CustomerOrderRespVO> getCustomerOrder(@RequestParam("id") Long id) {
        CustomerOrderDO customerOrder = customerOrderService.getCustomerOrder(id);
        return success(BeanUtils.toBean(customerOrder, CustomerOrderRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得客户总订单分页")
    @PreAuthorize("@ss.hasPermission('order:customer-order:query')")
    public CommonResult<PageResult<CustomerOrderRespVO>> getCustomerOrderPage(@Valid CustomerOrderPageReqVO pageReqVO) {
        PageResult<CustomerOrderDO> pageResult = customerOrderService.getCustomerOrderPage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, CustomerOrderRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出客户总订单 Excel")
    @PreAuthorize("@ss.hasPermission('order:customer-order:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportCustomerOrderExcel(@Valid CustomerOrderPageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<CustomerOrderDO> list = customerOrderService.getCustomerOrderPage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "客户总订单.xls", "数据", CustomerOrderRespVO.class,
                        BeanUtils.toBean(list, CustomerOrderRespVO.class));
    }

}