package cn.iocoder.foodnexus.module.order.controller.app.shoppingcart;

import cn.iocoder.foodnexus.framework.common.enums.UserSystemEnum;
import cn.iocoder.foodnexus.framework.common.util.CommonUtil;
import cn.iocoder.foodnexus.framework.common.util.collection.MapUtils;
import cn.iocoder.foodnexus.framework.security.core.util.SecurityFrameworkUtils;
import cn.iocoder.foodnexus.module.order.controller.app.shoppingcart.vo.ShoppingCartPageReqVO;
import cn.iocoder.foodnexus.module.order.controller.app.shoppingcart.vo.ShoppingCartRespVO;
import cn.iocoder.foodnexus.module.order.controller.app.shoppingcart.vo.ShoppingCartSaveReqVO;
import cn.iocoder.foodnexus.module.product.api.ProductSpuApi;
import cn.iocoder.foodnexus.module.product.api.dto.ProductInfo;
import cn.iocoder.foodnexus.module.product.dal.dataobject.spu.ProductSpuDO;
import cn.iocoder.foodnexus.module.product.service.spu.ProductSpuService;
import cn.iocoder.foodnexus.module.system.annotations.AppSystemAuth;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import jakarta.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import jakarta.validation.*;

import java.util.*;

import cn.iocoder.foodnexus.framework.common.pojo.PageResult;
import cn.iocoder.foodnexus.framework.common.pojo.CommonResult;
import cn.iocoder.foodnexus.framework.common.util.object.BeanUtils;
import static cn.iocoder.foodnexus.framework.common.pojo.CommonResult.success;

import cn.iocoder.foodnexus.module.order.controller.app.shoppingcart.vo.*;
import cn.iocoder.foodnexus.module.order.dal.dataobject.shoppingcart.ShoppingCartDO;
import cn.iocoder.foodnexus.module.order.service.shoppingcart.ShoppingCartService;

@Tag(name = "APP - 购物车")
@RestController
@RequestMapping("/order/shopping-cart")
@Validated
@AppSystemAuth(UserSystemEnum.CUSTOMER)
public class ShoppingCartController {

    @Resource
    private ShoppingCartService shoppingCartService;

    @Autowired
    private ProductSpuService productService;

    @PostMapping("/create")
    @Operation(summary = "创建购物车")
    public CommonResult<Long> createShoppingCart(@Valid @RequestBody ShoppingCartSaveReqVO createReqVO) {
        return success(shoppingCartService.createShoppingCart(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新购物车")
    public CommonResult<Boolean> updateShoppingCart(@Valid @RequestBody ShoppingCartSaveReqVO updateReqVO) {
        shoppingCartService.updateShoppingCart(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除购物车")
    @Parameter(name = "id", description = "编号", required = true)
    public CommonResult<Boolean> deleteShoppingCart(@RequestParam("id") Long id) {
        shoppingCartService.deleteShoppingCart(id);
        return success(true);
    }

    @DeleteMapping("/delete-list")
    @Parameter(name = "ids", description = "编号", required = true)
    @Operation(summary = "批量删除购物车")
    public CommonResult<Boolean> deleteShoppingCartList(@RequestParam("ids") List<Long> ids) {
        shoppingCartService.deleteShoppingCartListByIds(ids);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得购物车")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    public CommonResult<ShoppingCartRespVO> getShoppingCart(@RequestParam("id") Long id) {
        ShoppingCartDO shoppingCart = shoppingCartService.getShoppingCart(id);
        return success(BeanUtils.toBean(shoppingCart, ShoppingCartRespVO.class, cart ->
                cart.setProductInfo(BeanUtils.toBean(productService.getSpu(cart.getProductId()), ProductInfo.class))));
    }

    @GetMapping("/page")
    @Operation(summary = "获得购物车分页")
    public CommonResult<PageResult<ShoppingCartRespVO>> getShoppingCartPage(@Valid ShoppingCartPageReqVO pageReqVO) {
        pageReqVO.setCreator(SecurityFrameworkUtils.getLoginUserId());
        PageResult<ShoppingCartDO> pageResult = shoppingCartService.getShoppingCartPage(pageReqVO);
        Map<Long, ProductSpuDO> productMap = productService.getSpuMap(CommonUtil.listConvertSet(pageResult.getList(), ShoppingCartDO::getProductId));
        return success(BeanUtils.toBean(pageResult, ShoppingCartRespVO.class, item -> {
            MapUtils.findAndThen(productMap, item.getProductId(), product -> item.setProductInfo(BeanUtils.toBean(product, ProductInfo.class)));
        }));
    }

}