package cn.iocoder.foodnexus.module.order.controller.app.delivery;

import cn.iocoder.foodnexus.framework.common.enums.UserSystemEnum;
import cn.iocoder.foodnexus.framework.common.pojo.CommonResult;
import cn.iocoder.foodnexus.framework.common.pojo.PageParam;
import cn.iocoder.foodnexus.framework.common.pojo.PageResult;
import cn.iocoder.foodnexus.framework.common.util.CommonUtil;
import cn.iocoder.foodnexus.framework.common.util.object.BeanUtils;
import cn.iocoder.foodnexus.framework.common.validation.InEnum;
import cn.iocoder.foodnexus.framework.security.core.util.SecurityFrameworkUtils;
import cn.iocoder.foodnexus.module.erp.controller.admin.sale.vo.order.DeliveryMyCountRespVO;
import cn.iocoder.foodnexus.module.erp.controller.admin.sale.vo.order.DeliveryOrderPageReqVO;
import cn.iocoder.foodnexus.module.erp.controller.admin.sale.vo.order.ErpSaleOrderPageReqVO;
import cn.iocoder.foodnexus.module.erp.dal.dataobject.sale.ErpSaleOrderDO;
import cn.iocoder.foodnexus.module.erp.enums.SaleOrderPickUpStatus;
import cn.iocoder.foodnexus.module.erp.service.sale.ErpSaleOrderService;
import cn.iocoder.foodnexus.module.operations.service.deliverystaffcustomer.DeliveryStaffCustomerService;
import cn.iocoder.foodnexus.module.order.api.DeliveryStaffApi;
import cn.iocoder.foodnexus.module.order.controller.admin.customerorder.vo.CustomerOrderRespVO;
import cn.iocoder.foodnexus.module.order.controller.admin.customerorderitem.vo.CustomerOrderItemRespVO;
import cn.iocoder.foodnexus.module.order.controller.app.delivery.vo.DeliveryOrderDetails;
import cn.iocoder.foodnexus.module.order.controller.app.delivery.vo.DeliverySaleOrderRespVO;
import cn.iocoder.foodnexus.module.erp.controller.admin.sale.vo.order.DeliveryStatusCountRespVO;
import cn.iocoder.foodnexus.module.order.dto.DeliveryOrderUpdateReqVO;
import cn.iocoder.foodnexus.module.order.service.customerorder.CustomerOrderService;
import cn.iocoder.foodnexus.module.order.service.customerorderitem.CustomerOrderItemService;
import cn.iocoder.foodnexus.module.system.annotations.AppSystemAuth;
import cn.iocoder.foodnexus.module.system.controller.admin.dept.vo.dept.DeptSimpleRespVO;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Set;

import static cn.iocoder.foodnexus.framework.common.pojo.CommonResult.success;
import static cn.iocoder.foodnexus.framework.common.util.collection.CollectionUtils.convertMultiMap;
import static cn.iocoder.foodnexus.framework.common.util.collection.CollectionUtils.convertSet;
import static cn.iocoder.foodnexus.framework.security.core.util.SecurityFrameworkUtils.getLoginUserId;

/**
 * @author : yanghao
 * create at:  2025/10/14  17:56
 * @description: 配送员 - 订单
 */
@Tag(name = "配送员 - 订单")
@RestController
@RequestMapping("/delivery/sale-order")
@Validated
@AppSystemAuth(UserSystemEnum.DELIVERY)
public class DeliverySaleOrderController {

    @Autowired
    private DeliveryStaffApi deliveryStaffApi;

    @Autowired
    private DeliveryStaffCustomerService deliveryStaffCustomerService;

    @Autowired
    private ErpSaleOrderService saleOrderService;

    @Autowired
    private CustomerOrderService customerOrderService;

    @Autowired
    private CustomerOrderItemService customerOrderItemService;

    @GetMapping("/my-count")
    @Operation(summary = "个人中心 - 配送订单数")
    public CommonResult<DeliveryMyCountRespVO> queryMyCount() {
        return success(saleOrderService.queryMyCount(getLoginUserId()));
    }

    @GetMapping("/status-count")
    @Operation(summary = "接单大厅 - 配送状态订单数统计")
    public CommonResult<List<DeliveryStatusCountRespVO>> statusCount() {
        List<DeptSimpleRespVO> deptSimpleRespVOS = deliveryStaffCustomerService.queryDeptListByStaffId(deliveryStaffApi.queryStaffIdByUserId(getLoginUserId()));
        Set<Long> deptIds = CommonUtil.listConvertSet(deptSimpleRespVOS, DeptSimpleRespVO::getId);
        return success(saleOrderService.statusCount(getLoginUserId(), deptIds));
    }

    @GetMapping("/my-page")
    @Operation(summary = "个人中心 - 订单")
    public CommonResult<PageResult<DeliverySaleOrderRespVO>> queryMyPage(@Valid DeliveryOrderPageReqVO reqVO) {
        if (reqVO.getType() == 1) {
            reqVO.setDeliveryStaffId(deliveryStaffApi.queryStaffIdByUserId(getLoginUserId()));
            ErpSaleOrderPageReqVO pageReqVO = BeanUtils.toBean(reqVO, ErpSaleOrderPageReqVO.class);
            return success(BeanUtils.toBean(saleOrderService.getSaleOrderPage(pageReqVO), DeliverySaleOrderRespVO.class));
        }
        return success(new PageResult<>());
    }

    @GetMapping("/page")
    @Operation(summary = "接单大厅 - 查询配送单")
    @Parameter(name = "pickUpStatus", description = "拣货状态(字典类型:sale_order_pick_up_status)", required = true, example = "1024")
    public CommonResult<PageResult<DeliverySaleOrderRespVO>> getSaleOrderPage(@Valid PageParam pageReq,
                                                                              @InEnum(value = SaleOrderPickUpStatus.class)
                                                                              @RequestParam("pickUpStatus")
                                                                              String pickUpStatus) {
        ErpSaleOrderPageReqVO pageReqVO = new ErpSaleOrderPageReqVO();
        pageReqVO.setPageNo(pageReq.getPageNo());
        pageReqVO.setPageSize(pageReq.getPageSize());
        if (SaleOrderPickUpStatus.PICK_UP.getType().equals(pickUpStatus) || SaleOrderPickUpStatus.TO_BE.getType().equals(pickUpStatus) ) {
            List<DeptSimpleRespVO> deptSimpleRespVOS = deliveryStaffCustomerService.queryDeptListByStaffId(deliveryStaffApi.queryStaffIdByUserId(getLoginUserId()));
            pageReqVO.setCustomerDeptIds(CommonUtil.listConvert(deptSimpleRespVOS, DeptSimpleRespVO::getId));
        } else {
            pageReqVO.setDeliveryStaffId(deliveryStaffApi.queryStaffIdByUserId(getLoginUserId()));
        }
        pageReqVO.setPickUpStatus(SaleOrderPickUpStatus.getByType(pickUpStatus));
        return success(BeanUtils.toBean(saleOrderService.getSaleOrderPage(pageReqVO), DeliverySaleOrderRespVO.class));
    }

    @GetMapping("/get")
    @Operation(summary = "配送单详情")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    public CommonResult<DeliveryOrderDetails<DeliverySaleOrderRespVO>> getSaleOrder(@RequestParam("id") Long id) {
        ErpSaleOrderDO saleOrder = saleOrderService.getSaleOrder(id);
        if (saleOrder == null) {
            return success(null);
        }
        DeliveryOrderDetails<DeliverySaleOrderRespVO> result = new DeliveryOrderDetails<>();
        result.setDeliveryInfo(BeanUtils.toBean(saleOrder, DeliverySaleOrderRespVO.class));
        result.setCustomerOrderInfo(BeanUtils.toBean(customerOrderService.getCustomerOrder(saleOrder.getCustomerOrderId()),
                CustomerOrderRespVO.class, item ->
                        item.setOrderItems(BeanUtils.toBean(customerOrderItemService.queryByOrderId(item.getId()), CustomerOrderItemRespVO.class))));
        return success(result);
    }

    @PostMapping("/accept")
    @Operation(summary = "确认接单")
    @Parameter(name = "id", description = "配送单id(deliveryInfo.id)", required = true)
    public CommonResult<Boolean> accept(@RequestParam("id") Long id) {
        saleOrderService.accept(id, deliveryStaffApi.queryStaffIdByUserId(getLoginUserId()));
        return success(Boolean.TRUE);
    }

    // 配送订单，生成销售出库单
    @PostMapping("/delivery")
    @Operation(summary = "确认取货")
    @Parameter(name = "id", description = "配送单id(deliveryInfo.id)", required = true)
    public CommonResult<Boolean> delivery(@RequestParam("id") Long id) {
        saleOrderService.delivery(id, deliveryStaffApi.queryStaffIdByUserId(getLoginUserId()));
        return success(Boolean.TRUE);
    }

    // 订单到货
    @PostMapping("/arrival")
    @Operation(summary = "订单到货")
    public CommonResult<Boolean> arrival(@RequestBody @Valid DeliveryOrderUpdateReqVO reqVO) {
        saleOrderService.arrival(reqVO, deliveryStaffApi.queryStaffIdByUserId(getLoginUserId()));
        return success(Boolean.TRUE);
    }

}
