package cn.iocoder.foodnexus.module.operations.controller.admin.inquiresupplierpush;

import cn.iocoder.foodnexus.framework.common.util.CommonUtil;
import cn.iocoder.foodnexus.framework.common.util.collection.MapUtils;
import cn.iocoder.foodnexus.module.erp.api.service.ErpSupplierApi;
import cn.iocoder.foodnexus.module.erp.dal.dataobject.purchase.ErpSupplierDO;
import cn.iocoder.foodnexus.module.erp.dal.dataobject.sale.ErpCustomerDO;
import cn.iocoder.foodnexus.module.erp.service.purchase.ErpSupplierService;
import org.apache.commons.compress.utils.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import jakarta.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import jakarta.validation.constraints.*;
import jakarta.validation.*;
import jakarta.servlet.http.*;
import java.util.*;
import java.io.IOException;
import java.util.stream.Collectors;

import cn.iocoder.foodnexus.framework.common.pojo.PageParam;
import cn.iocoder.foodnexus.framework.common.pojo.PageResult;
import cn.iocoder.foodnexus.framework.common.pojo.CommonResult;
import cn.iocoder.foodnexus.framework.common.util.object.BeanUtils;
import static cn.iocoder.foodnexus.framework.common.pojo.CommonResult.success;

import cn.iocoder.foodnexus.framework.excel.core.util.ExcelUtils;

import cn.iocoder.foodnexus.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.foodnexus.framework.apilog.core.enums.OperateTypeEnum.*;

import cn.iocoder.foodnexus.module.operations.controller.admin.inquiresupplierpush.vo.*;
import cn.iocoder.foodnexus.module.operations.dal.dataobject.inquiresupplierpush.InquireSupplierPushDO;
import cn.iocoder.foodnexus.module.operations.service.inquiresupplierpush.InquireSupplierPushService;

@Tag(name = "管理后台 - 询价推送")
@RestController
@RequestMapping("/operations/inquire-supplier-push")
@Validated
public class InquireSupplierPushController {

    @Resource
    private InquireSupplierPushService inquireSupplierPushService;

    @Autowired
    private ErpSupplierApi supplierApi;

    @Autowired
    private ErpSupplierService supplierService;


    @PostMapping("/create")
    @Operation(summary = "创建询价推送")
    @PreAuthorize("@ss.hasPermission('operations:inquire-supplier-push:create')")
    public CommonResult<Long> createInquireSupplierPush(@Valid @RequestBody InquireSupplierPushSaveReqVO createReqVO) {
        return success(inquireSupplierPushService.createInquireSupplierPush(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新询价推送")
    @PreAuthorize("@ss.hasPermission('operations:inquire-supplier-push:update')")
    public CommonResult<Boolean> updateInquireSupplierPush(@Valid @RequestBody InquireSupplierPushSaveReqVO updateReqVO) {
        inquireSupplierPushService.updateInquireSupplierPush(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除询价推送")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('operations:inquire-supplier-push:delete')")
    public CommonResult<Boolean> deleteInquireSupplierPush(@RequestParam("id") Long id) {
        inquireSupplierPushService.deleteInquireSupplierPush(id);
        return success(true);
    }

    @DeleteMapping("/delete-list")
    @Parameter(name = "ids", description = "编号", required = true)
    @Operation(summary = "批量删除询价推送")
                @PreAuthorize("@ss.hasPermission('operations:inquire-supplier-push:delete')")
    public CommonResult<Boolean> deleteInquireSupplierPushList(@RequestParam("ids") List<Long> ids) {
        inquireSupplierPushService.deleteInquireSupplierPushListByIds(ids);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得询价推送")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('operations:inquire-supplier-push:query')")
    public CommonResult<InquireSupplierPushRespVO> getInquireSupplierPush(@RequestParam("id") Long id) {
        InquireSupplierPushDO inquireSupplierPush = inquireSupplierPushService.getInquireSupplierPush(id);
        Map<Long, String> nameMap = supplierApi.queryNameMapByIds(CommonUtil.asList(inquireSupplierPush.getSupplierId()));
        return success(BeanUtils.toBean(inquireSupplierPush, InquireSupplierPushRespVO.class, item ->
                MapUtils.findAndThen(nameMap, item.getSupplierId(), item::setSupplierName)));
    }

    @GetMapping("/query-supplier-list")
    @Operation(summary = "根据询价获取推送供应商")
    @Parameter(name = "inquirePriceId", description = "询价inquirePriceId", required = true, example = "1024")
    public CommonResult<List<ErpSupplierDO>> querySupplierList(@RequestParam("inquirePriceId") Long inquirePriceId) {
        List<InquireSupplierPushDO> inquireSupplierPushDOS = inquireSupplierPushService.queryByInquireId(inquirePriceId);
        if (CommonUtil.isEmpty(inquireSupplierPushDOS)) {
            return success(Lists.newArrayList());
        }
        return success(supplierService.getSupplierList(CommonUtil.listConvert(inquireSupplierPushDOS, InquireSupplierPushDO::getSupplierId)));
    }

    @GetMapping("/page")
    @Operation(summary = "获得询价推送分页")
    @PreAuthorize("@ss.hasPermission('operations:inquire-supplier-push:query')")
    public CommonResult<PageResult<InquireSupplierPushRespVO>> getInquireSupplierPushPage(@Valid InquireSupplierPushPageReqVO pageReqVO) {
        PageResult<InquireSupplierPushDO> pageResult = inquireSupplierPushService.getInquireSupplierPushPage(pageReqVO);
        Map<Long, String> nameMap = supplierApi.queryNameMapByIds(pageResult.getList().stream().map(InquireSupplierPushDO::getSupplierId).toList());
        return success(BeanUtils.toBean(pageResult, InquireSupplierPushRespVO.class, item ->
                MapUtils.findAndThen(nameMap, item.getSupplierId(), item::setSupplierName)));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出询价推送 Excel")
    @PreAuthorize("@ss.hasPermission('operations:inquire-supplier-push:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportInquireSupplierPushExcel(@Valid InquireSupplierPushPageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<InquireSupplierPushDO> list = inquireSupplierPushService.getInquireSupplierPushPage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "询价推送.xls", "数据", InquireSupplierPushRespVO.class,
                        BeanUtils.toBean(list, InquireSupplierPushRespVO.class));
    }

}